<?php


if (session_status() === PHP_SESSION_NONE) { session_start(); }

// Reject non-POST quickly
if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
    header('Location: /novademo/html/demo1/index.php#cservices');
    exit;
}

// ---- Load DB config (../config.php) ----
$__cfg = dirname(__DIR__) . '/config.php';
if (!is_file($__cfg)) {
    $_SESSION['cservices_flash']['error'][] = 'Server config missing.';
    session_write_close();
    header('Location: /novademo/html/demo1/index.php#cservices');
    exit;
}
require_once $__cfg;

// ---- DB handle (prefer $pdo from config.php) ----
$pdo = $pdo ?? null;
if (!$pdo) {
    $dbHost = defined('DB_HOST') ? DB_HOST : null;
    $dbName = defined('DB_NAME') ? DB_NAME : null;
    $dbUser = defined('DB_USER') ? DB_USER : null;
    $dbPass = defined('DB_PASS') ? DB_PASS : null;
    try {
        if ($dbHost && $dbName && $dbUser !== null) {
            $pdo = new PDO(
                "mysql:host={$dbHost};dbname={$dbName};charset=utf8mb4",
                $dbUser,
                $dbPass ?? '',
                [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                ]
            );
        } else {
            throw new RuntimeException('Database config missing.');
        }
    } catch (Throwable $e) {
        $_SESSION['cservices_flash']['error'][] = 'DB connection failed.';
        session_write_close();
        header('Location: /novademo/html/demo1/index.php#cservices');
        exit;
    }
}

// ---- Auth ----
$currentUserId = $_SESSION['user_id'] ?? null;
if (!$currentUserId) {
    http_response_code(403);
    echo "Forbidden";
    exit;
}

// ---- Helpers ----
function flash_push(string $type, string $msg): void {
    if (!isset($_SESSION['cservices_flash'])) {
        $_SESSION['cservices_flash'] = ['success'=>[], 'error'=>[]];
    }
    $_SESSION['cservices_flash'][$type][] = $msg;
}
function sanitize_return_to(?string $rt): string {
    $fallback = '/novademo/html/demo1/index.php#cservices';
    if (!$rt) return $fallback;
    if (!preg_match('~^/[^\\r\\n]*$~', $rt)) return $fallback;
    return $rt;
}
function safe_redirect(string $url): void {
    session_write_close();
    if (!headers_sent()) {
        header("Location: " . $url);
        exit;
    }
    echo '<!doctype html><meta http-equiv="refresh" content="0;url=' . htmlspecialchars($url, ENT_QUOTES) . '">';
    echo '<a href="' . htmlspecialchars($url, ENT_QUOTES) . '">Continue</a>';
    exit;
}


function normalize_datetime_local(?string $val): ?string {
    if (!$val) return null;
    $val = trim($val);
    if ($val === '') return null;

    $val = str_replace('T', ' ', $val);

    if (preg_match('/^\d{4}-\d{2}-\d{2} \d{2}:\d{2}$/', $val)) {
        $val .= ':00';
    }

    $dt = DateTime::createFromFormat('Y-m-d H:i:s', $val);
    if (!$dt) return null;

    return $dt->format('Y-m-d H:i:s');
}

// ---- CSRF ----
$post_token = $_POST['csrf_token'] ?? '';
if (empty($_SESSION['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $post_token)) {
    flash_push('error', 'Invalid request token.');
    safe_redirect(sanitize_return_to($_POST['return_to'] ?? null));
}

$action   = $_POST['action'] ?? '';
$returnTo = sanitize_return_to($_POST['return_to'] ?? null);

$_SESSION['cservices_old'] = [
    'service_id' => $_POST['service_id'] ?? '',
    'start_at'   => $_POST['start_at'] ?? '',
    'end_at'     => $_POST['end_at'] ?? '',
];

try {
    if ($action === 'add') {
        // Inputs
        $service_id = isset($_POST['service_id']) ? (int)$_POST['service_id'] : 0;

        // start_at -> NOW (server)
        $start_val  = date('Y-m-d H:i:s');

        // end_at -> MANDATORY; normalize
        $raw_end    = $_POST['end_at'] ?? null;
        $end_val    = normalize_datetime_local($raw_end);
        if ($end_val === null) {
            flash_push('error', 'End date/time is required. Please choose a valid value.');
            safe_redirect($returnTo);
        }

        // Validate service selection
        if ($service_id <= 0) {
            flash_push('error', 'Please select a service.');
            safe_redirect($returnTo);
        }

        // Service must exist and be active
        $st = $pdo->prepare("SELECT id FROM services WHERE id = :id AND is_active = 1");
        $st->execute([':id' => $service_id]);
        if (!$st->fetch()) {
            flash_push('error', 'Selected service is not available.');
            safe_redirect($returnTo);
        }

        // Duplicate (any status)
        $dup = $pdo->prepare("SELECT id FROM client_services WHERE client_user_id = :u AND service_id = :s LIMIT 1");
        $dup->execute([':u' => $currentUserId, ':s' => $service_id]);
        if ($dup->fetch()) {
            flash_push('error', 'This service already exists for your account.');
            safe_redirect($returnTo);
        }

        $ins = $pdo->prepare("
            INSERT INTO client_services
                (client_user_id, service_id, start_at, end_at, assigned_by_user_id, is_active)
            VALUES
                (:uid, :sid, :start_at, :end_at, :assigned_by, 0)
        ");
        $ins->execute([
            ':uid'         => $currentUserId,
            ':sid'         => $service_id,
            ':start_at'    => $start_val,
            ':end_at'      => $end_val,
            ':assigned_by' => $currentUserId,
        ]);

        unset($_SESSION['cservices_old']);
        flash_push('success', 'Your service request was submitted and is pending admin approval.');
        safe_redirect($returnTo);

    } else {
        flash_push('error', 'Unknown action.');
        safe_redirect($returnTo);
    }

} catch (Throwable $e) {
    flash_push('error', 'Operation failed.');
    safe_redirect($returnTo);
}
