<?php
/**
 * Handler: Add/Update/Start-Edit/Cancel-Edit for client assets
 * Path: /novademo/html/demo1/userdashboard/assets_handler.php
 * Redirects back to /novademo/html/demo1/index.php#assets with flash messages and/or edit state.
 */

if (session_status() === PHP_SESSION_NONE) { session_start(); }

// ---- Load DB config ----
$__cfg = dirname(__DIR__) . '/config.php';
if (!is_file($__cfg)) { die('config.php not found at /novademo/html/demo1/config.php'); }
require_once $__cfg;

// ---- DB handle ----
$pdo = $pdo ?? null;
if (!$pdo) {
    $dbHost = defined('DB_HOST') ? DB_HOST : null;
    $dbName = defined('DB_NAME') ? DB_NAME : null;
    $dbUser = defined('DB_USER') ? DB_USER : null;
    $dbPass = defined('DB_PASS') ? DB_PASS : null;
    if ($dbHost && $dbName && $dbUser !== null) {
        try {
            $pdo = new PDO(
                "mysql:host={$dbHost};dbname={$dbName};charset=utf8mb4",
                $dbUser,
                $dbPass ?? '',
                [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                ]
            );
        } catch (Throwable $e) {
            $_SESSION['asset_flash']['error'][] = "DB connection failed.";
            header('Location: /novademo/html/demo1/index.php#assets'); exit;
        }
    } else {
        $_SESSION['asset_flash']['error'][] = "Database config missing.";
        header('Location: /novademo/html/demo1/index.php#assets'); exit;
    }
}

// ---- Utilities ----
function redirect_back(string $to = '/novademo/html/demo1/index.php#assets') {
    header('Location: ' . $to);
    exit;
}
function assetRowOwned(PDO $pdo, int $uid, int $id): ?array {
    $st = $pdo->prepare("SELECT ca.id, ca.asset_id, ca.amount, ca.is_active, ca.updated_at,
                                a.symbol, a.name, a.decimals
                           FROM client_assets ca
                           JOIN assets a ON a.id = ca.asset_id
                          WHERE ca.id = :id AND ca.client_user_id = :u
                          LIMIT 1");
    $st->execute([':id'=>$id, ':u'=>$uid]);
    $r = $st->fetch();
    return $r ?: null;
}
function assetExistsAnyOther(PDO $pdo, int $uid, int $assetId, int $excludeId = 0): bool {
    $sql = "SELECT 1 FROM client_assets WHERE client_user_id=:u AND asset_id=:a";
    $params = [':u'=>$uid, ':a'=>$assetId];
    if ($excludeId > 0) { $sql .= " AND id <> :id"; $params[':id'] = $excludeId; }
    $sql .= " LIMIT 1";
    $st = $pdo->prepare($sql);
    $st->execute($params);
    return (bool)$st->fetchColumn();
}

// ---- Auth ----
$currentUserId = $_SESSION['user_id'] ?? null;
if (!$currentUserId) {
    $_SESSION['asset_flash']['error'][] = "You must be logged in.";
    redirect_back();
}

// ---- Input ----
$action     = $_POST['action']     ?? '';
$return_to  = $_POST['return_to']  ?? '/novademo/html/demo1/index.php#assets';
$csrfPost   = (string)($_POST['csrf_token'] ?? '');
$csrfSess   = (string)($_SESSION['csrf_token'] ?? '');

if ($csrfPost === '' || $csrfSess === '' || !hash_equals($csrfSess, $csrfPost)) {
    $_SESSION['asset_flash']['error'][] = "Security token mismatch. Please try again.";
    redirect_back($return_to);
}

switch ($action) {
    case 'start_edit': {
        $id = isset($_POST['client_asset_id']) ? (int)$_POST['client_asset_id'] : 0;
        if ($id <= 0) {
            $_SESSION['asset_flash']['error'][] = "Invalid edit request.";
            redirect_back($return_to);
        }
        $row = assetRowOwned($pdo, $currentUserId, $id);
        if (!$row) {
            $_SESSION['asset_flash']['error'][] = "This asset entry does not exist or does not belong to you.";
            redirect_back($return_to);
        }
        // Store edit state in session (read by assets.php)
        $_SESSION['asset_edit'] = [
            'id'       => (int)$row['id'],
            'asset_id' => (int)$row['asset_id'],
            'amount'   => (string)$row['amount'],
            'decimals' => isset($row['decimals']) ? (int)$row['decimals'] : null,
        ];
        redirect_back($return_to);
    }

    case 'cancel_edit': {
        unset($_SESSION['asset_edit'], $_SESSION['asset_old']);
        redirect_back($return_to);
    }

    case 'add': {
        $asset_id  = isset($_POST['asset_id']) ? (int)$_POST['asset_id'] : 0;
        $amountRaw = trim($_POST['amount'] ?? '');
        $_SESSION['asset_old'] = ['asset_id'=>(string)$asset_id, 'amount'=>$amountRaw];

        // Validate
        $errors = [];
        if ($asset_id <= 0) { $errors[] = "Please select an asset."; }
        if ($amountRaw === '' || !is_numeric($amountRaw)) { $errors[] = "Please enter a valid amount."; }

        if ($asset_id > 0) {
            $st = $pdo->prepare("SELECT id FROM assets WHERE id = :id AND is_active = 1 LIMIT 1");
            $st->execute([':id'=>$asset_id]);
            if (!$st->fetch()) { $errors[] = "Selected asset is not available."; }
        }

        if ($errors) {
            foreach ($errors as $e) $_SESSION['asset_flash']['error'][] = $e;
            redirect_back($return_to);
        }

        // Prevent duplicates
        if (assetExistsAnyOther($pdo, $currentUserId, $asset_id, 0)) {
            $_SESSION['asset_flash']['error'][] = "The asset already exists in your list.";
            redirect_back($return_to);
        }

        $ins = $pdo->prepare("INSERT INTO client_assets
            (client_user_id, asset_id, amount, last_updated_by_user_id, updated_at, is_active)
            VALUES (:u, :a, :amt, :updby, NOW(), 0)");
        $ins->execute([
            ':u'     => $currentUserId,
            ':a'     => $asset_id,
            ':amt'   => $amountRaw,
            ':updby' => $currentUserId,
        ]);

        unset($_SESSION['asset_old']);
        $_SESSION['asset_flash']['success'][] = "Asset request submitted and is pending activation by admin.";
        redirect_back($return_to);
    }

    case 'update': {
        // Must have an edit state or at least a valid id from form
        $client_asset_id = isset($_POST['client_asset_id']) ? (int)$_POST['client_asset_id'] : 0;
        if ($client_asset_id <= 0 && !empty($_SESSION['asset_edit']['id'])) {
            $client_asset_id = (int)$_SESSION['asset_edit']['id'];
        }
        if ($client_asset_id <= 0) {
            $_SESSION['asset_flash']['error'][] = "Invalid update request.";
            redirect_back($return_to);
        }

        // Ensure record belongs to user
        $row = assetRowOwned($pdo, $currentUserId, $client_asset_id);
        if (!$row) {
            $_SESSION['asset_flash']['error'][] = "This asset entry does not exist or does not belong to you.";
            redirect_back($return_to);
        }

        $asset_id  = isset($_POST['asset_id']) ? (int)$_POST['asset_id'] : 0;
        $amountRaw = trim($_POST['amount'] ?? '');
        $_SESSION['asset_old'] = ['asset_id'=>(string)$asset_id, 'amount'=>$amountRaw];

        // Validate
        $errors = [];
        if ($asset_id <= 0) { $errors[] = "Please select an asset."; }
        if ($amountRaw === '' || !is_numeric($amountRaw)) { $errors[] = "Please enter a valid amount."; }

        if ($asset_id > 0) {
            $st = $pdo->prepare("SELECT id FROM assets WHERE id = :id AND is_active = 1 LIMIT 1");
            $st->execute([':id'=>$asset_id]);
            if (!$st->fetch()) { $errors[] = "Selected asset is not available."; }
        }

        if ($errors) {
            foreach ($errors as $e) $_SESSION['asset_flash']['error'][] = $e;
            redirect_back($return_to);
        }

        // Avoid duplicates if changing asset_id
        if (assetExistsAnyOther($pdo, $currentUserId, $asset_id, $client_asset_id)) {
            $_SESSION['asset_flash']['error'][] = "Another entry for this asset already exists.";
            redirect_back($return_to);
        }

        $upd = $pdo->prepare("UPDATE client_assets
                                 SET asset_id = :a,
                                     amount = :amt,
                                     last_updated_by_user_id = :updby,
                                     updated_at = NOW(),
                                     is_active = 0
                               WHERE id = :id AND client_user_id = :u");
        $upd->execute([
            ':a'     => $asset_id,
            ':amt'   => $amountRaw,
            ':updby' => $currentUserId,
            ':id'    => $client_asset_id,
            ':u'     => $currentUserId,
        ]);

        unset($_SESSION['asset_old'], $_SESSION['asset_edit']);
        $_SESSION['asset_flash']['success'][] = "Asset updated and set to inactive for admin review.";
        redirect_back($return_to);
    }

    default: {
        $_SESSION['asset_flash']['error'][] = "Unknown action.";
        redirect_back($return_to);
    }
}
