<?php


if (session_status() === PHP_SESSION_NONE) { session_start(); }

// ---- Load DB config (../config.php) ----
$__cfg = dirname(__DIR__) . '/config.php';
if (!is_file($__cfg)) { die('config.php not found at /novademo/html/demo1/config.php'); }
require_once $__cfg;

// ---- DB handle (prefer $pdo from config.php) ----
$pdo = $pdo ?? null;
if (!$pdo) {
    $dbHost = defined('DB_HOST') ? DB_HOST : null;
    $dbName = defined('DB_NAME') ? DB_NAME : null;
    $dbUser = defined('DB_USER') ? DB_USER : null;
    $dbPass = defined('DB_PASS') ? DB_PASS : null;
    if ($dbHost && $dbName && $dbUser !== null) {
        try {
            $pdo = new PDO(
                "mysql:host={$dbHost};dbname={$dbName};charset=utf8mb4",
                $dbUser,
                $dbPass ?? '',
                [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                ]
            );
        } catch (Throwable $e) {
            die("DB connection failed: " . htmlspecialchars($e->getMessage()));
        }
    } else {
        die("Database config missing. Provide \$pdo or DB_* constants in config.php.");
    }
}

// ---- Auth ----
$currentUserId = $_SESSION['user_id'] ?? null;
if (!$currentUserId) {
    ?>
    <div id="page-content" class="w-full flex justify-center py-12">
      <div class="kt-card max-w-[600px] w-full">
        <div class="kt-card-content flex flex-col gap-5 p-10">
          <h2 class="text-xl font-semibold text-center">Client Services</h2>
          <div class="text-sm p-4 rounded-lg bg-accent/40 text-secondary-foreground text-center">
            You must be logged in to manage your services.
            <a href="#login" data-page="login" class="text-primary underline ml-1">Go to Login</a>
          </div>
        </div>
      </div>
    </div>
    <?php
    return;
}

// ---- CSRF ----
if (empty($_SESSION['csrf_token'])) { $_SESSION['csrf_token'] = bin2hex(random_bytes(16)); }
$csrf_token = $_SESSION['csrf_token'];

$flash = $_SESSION['cservices_flash'] ?? ['success'=>[], 'error'=>[]];
unset($_SESSION['cservices_flash']);

$old   = $_SESSION['cservices_old'] ?? ['service_id'=>'', 'start_at'=>'', 'end_at'=>''];
unset($_SESSION['cservices_old']);

// ---- Helpers ----
function fetchActiveServicesForDropdown(PDO $pdo): array {
    return $pdo->query("
        SELECT id, description
          FROM services
         WHERE is_active = 1
      ORDER BY description ASC
    ")->fetchAll();
}

function fetchClientServicesAll(PDO $pdo, int $uid): array {
    $st = $pdo->prepare("
        SELECT cs.id, cs.service_id, cs.start_at, cs.end_at, cs.is_active, cs.assigned_by_user_id,
               s.description
          FROM client_services cs
          JOIN services s ON s.id = cs.service_id
         WHERE cs.client_user_id = :u
      ORDER BY s.description, cs.id
    ");
    $st->execute([':u'=>$uid]);
    return $st->fetchAll();
}

function formatDateTimeNullable(?string $d): string {
    if (!$d) return '';
    $ts = strtotime($d);
    return $ts ? date('Y-m-d H:i', $ts) : htmlspecialchars($d);
}

function isValidAndNotPast(?string $endAt): bool {
    if ($endAt === null || $endAt === '') return false;
    $ts = strtotime($endAt);
    if ($ts === false) return false;
    return $ts >= time();
}

// ---- Data for dropdown and lists ----
$dropdownServices = fetchActiveServicesForDropdown($pdo);
$clientAll        = fetchClientServicesAll($pdo, $currentUserId);

$clientActive = [];
$clientOther  = []; // with reason

foreach ($clientAll as $row) {
    $isActiveFlag = (int)$row['is_active'] === 1;
    $endOK        = isValidAndNotPast($row['end_at']);

    if ($isActiveFlag && $endOK) {
        $clientActive[] = $row;
    } else {
        $reason = 'Inactive';
        $hasEnd = ($row['end_at'] !== null && $row['end_at'] !== '');
        $endTs  = $hasEnd ? strtotime($row['end_at']) : false;

        if ($hasEnd && $endTs !== false && $endTs < time()) {
            $reason = 'Expired';
        } elseif ((int)$row['is_active'] === 0) {
            $reason = 'Pending';
        } else {
            if (!$hasEnd) {
                $reason = 'Missing end';
            } elseif ($endTs === false) {
                $reason = 'Invalid end';
            }
        }

        $row['_reason'] = $reason;
        $clientOther[]  = $row;
    }
}

// ---- URLs ----
$BASE_URL     = '/novademo/html/demo1';
$HANDLER_URL  = $BASE_URL . '/userdashboard/cservices_handler.php';
$RETURN_TO    = $BASE_URL . '/index.php#cservices';

// ---- Prefill values for inputs
$nowInputValue = date('Y-m-d\\TH:i');
$startInputVal = $nowInputValue; // always show now; server sets real value
$endInputVal   = $old['end_at']; // keep raw posted value on error
?>

<!-- Client Services Section -->
<section id="cservices" class="py-8" style="width:70%;max-width:1100px;margin:0 auto;">
  <div class="kt-card">
    <div class="kt-card-content p-10 flex flex-col gap-5">
      <div class="text-center">
        <h3 class="text-lg font-medium text-mono">Manage Your Services</h3>
        <span class="text-sm text-secondary-foreground">Request new services or review your current ones</span>
      </div>

      <!-- Flash messages -->
      <?php foreach (($flash['success'] ?? []) as $m): ?>
        <div class="kt-form-label text-green-600" role="status"><?= htmlspecialchars($m) ?></div>
      <?php endforeach; ?>
      <?php foreach (($flash['error'] ?? []) as $m): ?>
        <div class="kt-form-label text-red-600" role="alert"><?= htmlspecialchars($m) ?></div>
      <?php endforeach; ?>

      <!-- Request Service Form -->
      <form id="cservices_form" class="flex flex-col gap-5" method="post" action="<?= htmlspecialchars($HANDLER_URL) ?>">
        <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($csrf_token) ?>">
        <input type="hidden" name="return_to" value="<?= htmlspecialchars($RETURN_TO) ?>">
        <input type="hidden" name="action" value="add">

        <div class="flex flex-col gap-1">
          <label class="kt-form-label font-normal text-mono" for="service_id">Service</label>
          <select id="service_id" name="service_id" class="kt-input" required>
            <option value="">-- Select Service --</option>
            <?php foreach ($dropdownServices as $s): ?>
              <?php $sel = ($old['service_id'] !== '' && (int)$old['service_id'] === (int)$s['id']) ? 'selected' : ''; ?>
              <option value="<?= (int)$s['id'] ?>" <?= $sel ?>>
                <?= htmlspecialchars($s['description']) ?>
              </option>
            <?php endforeach; ?>
          </select>
        </div>

        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div class="flex flex-col gap-1">
            <label class="kt-form-label font-normal text-mono" for="start_at">Start (auto, now)</label>
            <input id="start_at" name="start_at" type="datetime-local" class="kt-input"
                   value="<?= htmlspecialchars($nowInputValue) ?>" readonly>
          </div>
          <div class="flex flex-col gap-1">
            <label class="kt-form-label font-normal text-mono" for="end_at">End (required)</label>
            <input id="end_at" name="end_at" type="datetime-local" class="kt-input"
                   value="<?= htmlspecialchars($endInputVal) ?>" required
                   min="<?= htmlspecialchars($nowInputValue) ?>">
          </div>
        </div>

        <div class="flex items-center gap-3">
          <button type="submit" class="kt-btn kt-btn-primary">
            Request<i class="ki-filled ki-black-right"></i>
          </button>
        </div>
      </form>

      <!-- Active Services -->
      <div class="mt-4">
        <h4 class="text-base font-semibold mb-3">Your Active Services</h4>
        <?php if (count($clientActive) === 0): ?>
          <div class="text-sm text-secondary-foreground">No active services found.</div>
        <?php else: ?>
          <ul class="space-y-2" id="services-active-list">
            <?php foreach ($clientActive as $row): ?>
              <li class="flex items-center justify-between p-3 rounded-md border border-border">
                <div>
                  <div class="font-medium"><?= htmlspecialchars($row['description']) ?></div>
                  <div class="text-sm text-secondary-foreground">
                    Start: <?= htmlspecialchars(formatDateTimeNullable($row['start_at'])) ?>,
                    End: <?= htmlspecialchars(formatDateTimeNullable($row['end_at'])) ?>
                  </div>
                </div>
              </li>
            <?php endforeach; ?>
          </ul>
        <?php endif; ?>
      </div>

      <!-- Pending / Expired / Invalid -->
      <?php if (count($clientOther) > 0): ?>
        <div class="mt-6">
          <h4 class="text-base font-semibold mb-3">Your Pending / Expired Services</h4>
          <ul class="space-y-2" id="services-inactive-list">
            <?php foreach ($clientOther as $row): ?>
              <li class="flex items-center justify-between p-3 rounded-md border border-border">
                <div>
                  <div class="font-medium">
                    <?= htmlspecialchars($row['description']) ?>
                    <span class="ml-2 text-xs opacity-70">(<?= htmlspecialchars($row['_reason']) ?>)</span>
                  </div>
                  <div class="text-sm text-secondary-foreground">
                    Start: <?= htmlspecialchars(formatDateTimeNullable($row['start_at'])) ?>,
                    End: <?= htmlspecialchars(formatDateTimeNullable($row['end_at'])) ?>
                  </div>
                </div>
              </li>
            <?php endforeach; ?>
          </ul>
        </div>
      <?php endif; ?>
    </div>
  </div>
</section>
